#define COPYRIGHTINFO "\
PILc2g, an advanced color-to-gray transformation library for PIL\n\
By Oliver Siemoneit, copyright 2008\n\
oliver.siemoneit@web.de\n\
\n\
This library is free software; you can redistribute it and/or\n\
modify it under the terms of the GNU Lesser General Public\n\
License as published by the Free Software Foundation; either\n\
version 2.1 of the License, or (at your option) any later version.\n\
\n\
This library is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU\n\
Lesser General Public License for more details.\n\
\n\
You should have received a copy of the GNU Lesser General Public\n\
License along with this library; if not, write to the Free Software\n\
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA\n\
"

/////////////////////////////////////////////////////////////////////////////
// includes
/////////////////////////////////////////////////////////////////////////////
#include "Python.h"
#include "Imaging.h"


/////////////////////////////////////////////////////////////////////////////
// version information: update this before compiling for the versions you're using
/////////////////////////////////////////////////////////////////////////////
#define PILC2GVERSION       "0.1.0"
#define PILVERSION          "1.1.6"
#define PYTHONVERSION       "2.5.0"

/////////////////////////////////////////////////////////////////////////////
// version history
/////////////////////////////////////////////////////////////////////////////
/*
0.1.0   initial release
*/

/////////////////////////////////////////////////////////////////////////////
// known to-do list with current version
/////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////
// options / configuration
/////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////
// reference
/////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////
// structs
/////////////////////////////////////////////////////////////////////////////
typedef struct {
    PyObject_HEAD
    Imaging image;
} ImagingObject;


/////////////////////////////////////////////////////////////////////////////
// internal functions
/////////////////////////////////////////////////////////////////////////////

#ifndef max
   #define max( a, b ) ( ((a) > (b)) ? (a) : (b) )
#endif

#ifndef min
   #define min( a, b ) ( ((a) < (b)) ? (a) : (b) )
#endif


void
_rgb2hsl(UINT8 red, UINT8 green, UINT8 blue,
		 float &H, float &S, float &L) {
   // Code for rgb2hsl taken from http://www.easyrgb.com/math.php?MATH=M18#text18
   //
   // Faust's color2gray-algorithm expects returned values to be within range
   //     <0;1> for saturation and lightness, <0;360> for hue.
   float var_R = red / 255.0; 
   float var_G = green / 255.0;
   float var_B = blue / 255.0;

   float var_Min = min(var_R, min (var_G, var_B));    
   float var_Max = max(var_R, max (var_G, var_B));    
   float del_Max = var_Max - var_Min;                 

   L = (var_Max + var_Min) / 2;

   if (del_Max == 0) {                     
      H = 0;                                
      S = 0;
   }
   else {                                   
      if (L < 0.5)
         S = del_Max / ( var_Max + var_Min );
      else
         S = del_Max / ( 2 - var_Max - var_Min );

      float del_R = ( ( ( var_Max - var_R ) / 6 ) + ( del_Max / 2 ) ) / del_Max;
      float del_G = ( ( ( var_Max - var_G ) / 6 ) + ( del_Max / 2 ) ) / del_Max;
      float del_B = ( ( ( var_Max - var_B ) / 6 ) + ( del_Max / 2 ) ) / del_Max;

      if (var_R == var_Max)
         H = del_B - del_G;
      else if (var_G == var_Max)
         H = ( 1 / 3 ) + del_R - del_B;
      else if (var_B == var_Max)
         H = ( 2 / 3 ) + del_G - del_R;

     if (H < 0)
         H += 1;
     if (H > 1)
         H -= 1;
     H *= 360.0;
   }
}


int 
_c2g(Imaging im, Imaging imOut, int brightness_correction) {
  // Color2Gray image correction algorithm implemented according to
  // http://www.e56.de/c2g.php, copyrights by Martin Faust

  Py_BEGIN_ALLOW_THREADS 

  float *gray = NULL;
  gray = new float [im->xsize * im->ysize * 3];
  
  if (gray == NULL) {
    return -1;
  }

  float min_v = 100.0f;
  float max_v = -100.0f;
  float mean = 0.0f;
  UINT8 *pixel;
  float hue, saturation, brightness;
  
  for(int y=0; y<im->ysize; y++) {
     for(int x=0; x<im->xsize; x++) {
        pixel = (UINT8*) &(im->image32[y][x]);
        _rgb2hsl(pixel[0], pixel[1], pixel[2], hue, saturation, brightness);
        if (saturation == 0.0f)
	     gray[(y * im->xsize) + x] = 1.5f * brightness;
	  else
	     gray[(y * im->xsize) + x] = brightness +  brightness*saturation;

        if (gray[(y * im->xsize) + x] < min_v)
           min_v = gray[(y * im->xsize) + x];
        
        if (gray[(y * im->xsize) + x]  > max_v)
           max_v = gray[(y * im->xsize) + x];
	  
        mean += gray[(y * im->xsize) + x];
	}
  }
			
  mean /= (float) (im->ysize * im->xsize);
  min_v = 0.0f;
  max_v = (mean + max_v) * 0.5f;
	
  for(int y=0; y<im->ysize; y++) {
     for(int x=0; x<im->xsize; x++) {
        if (brightness_correction != 0)
	      brightness = 0.9f * 255.0f * (gray[(y * im->xsize) + x] - min_v) / (max_v - min_v);
	  else
		brightness = 255.0f * (gray[(y * im->xsize) + x] - min_v) / (max_v - min_v);
       
        if (brightness > 255.0f)
            brightness = 255.0f;

	  if (brightness < 0.0f)
            brightness = 0.0f;
	  
        imOut->image32[y][x] = (UINT8)brightness | (UINT8)brightness << 8 | (UINT8)brightness << 16 | (UINT8)brightness << 24;
     }
  }

  free(gray);
  
  Py_END_ALLOW_THREADS
 

  // return 0 on success
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
// Python callable functions
/////////////////////////////////////////////////////////////////////////////
static PyObject *
versions (PyObject *self, PyObject *args) {
  return Py_BuildValue ("sss", PILC2GVERSION, PYTHONVERSION, PILVERSION);
}

static PyObject *
about (PyObject *self, PyObject *args) {
  return Py_BuildValue("s", COPYRIGHTINFO);
}

static PyObject *
copyright (PyObject *self, PyObject *args) {
  return about(self, args);
}

static PyObject *
c2g (PyObject *self, PyObject *args) {
  long idIn = 0L;
  long idOut = 0L;

  Imaging im = NULL;
  Imaging imOut = NULL;

  int brightness_correction = 0;
  int result = 0;

  if (!PyArg_ParseTuple(args, "lli", &idIn, &idOut, &brightness_correction)) {
    return Py_BuildValue("is", -1, "ERROR: Could not parse argument tuple, check documentation.");
  }

  im = (Imaging) idIn;
  imOut = (Imaging) idOut;

  // Only RGB images supported for now
  if (strcmp(im->mode, "RGB") != 0)
    return Py_BuildValue("is", -1, "ERROR: only RGB images supported.");

  result = _c2g(im, imOut, brightness_correction);

  if (result == 0)
    return Py_BuildValue("is", 0, "");
  else
    return Py_BuildValue("is", result, "ERROR: unknown error occurred during c2g operation.");

}

/////////////////////////////////////////////////////////////////////////////
// Python interface setup
/////////////////////////////////////////////////////////////////////////////
static PyMethodDef PILc2g_methods[] = {
  // pyCMS info
  {"versions", versions, 1, ".versions()"},
  {"about", about, 1, ".about()"},
  {"copyright", copyright, 1, ".copyright()"},

  {"c2g", c2g, 1, ".c2g(imIn.im.id, imOut.im.id, radius)"},

  {NULL, NULL}
};


extern "C" {
  void initPILc2g(void)
  {
    Py_InitModule("PILc2g", PILc2g_methods);
  }
}
