#define COPYRIGHTINFO "\
PILdaltonize, a PIL image transformation library for color vision defecienies\n\
By Oliver Siemoneit, copyright 2008\n\
oliver.siemoneit@web.de\n\
\n\
This library is free software; you can redistribute it and/or\n\
modify it under the terms of the GNU Lesser General Public\n\
License as published by the Free Software Foundation; either\n\
version 2.1 of the License, or (at your option) any later version.\n\
\n\
This library is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU\n\
Lesser General Public License for more details.\n\
\n\
You should have received a copy of the GNU Lesser General Public\n\
License along with this library; if not, write to the Free Software\n\
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA\n\
"

/////////////////////////////////////////////////////////////////////////////
// includes
/////////////////////////////////////////////////////////////////////////////
#include "Python.h"
#include "Imaging.h"


/////////////////////////////////////////////////////////////////////////////
// version information: update this before compiling for the versions you're using
/////////////////////////////////////////////////////////////////////////////
#define PILDALTONIZE        "0.1.0"
#define PILVERSION          "1.1.6"
#define PYTHONVERSION       "2.5.0"

/////////////////////////////////////////////////////////////////////////////
// version history
/////////////////////////////////////////////////////////////////////////////
/*
0.1.0   initial release
*/

/////////////////////////////////////////////////////////////////////////////
// known to-do list with current version
/////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////
// options / configuration
/////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////
// reference
/////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////
// structs
/////////////////////////////////////////////////////////////////////////////
typedef struct {
    PyObject_HEAD
    Imaging image;
} ImagingObject;


/////////////////////////////////////////////////////////////////////////////
// internal functions
/////////////////////////////////////////////////////////////////////////////

#ifndef max
   #define max( a, b ) ( ((a) > (b)) ? (a) : (b) )
#endif

#ifndef min
   #define min( a, b ) ( ((a) < (b)) ? (a) : (b) )
#endif

void
_initArray (float target[3][3], float source[3][3]) {
  for (int i=0; i<3; i++) {
    for (int j=0; j<3; j++)
      target[i][j]=source[i][j];
  }
}

inline void
_dot (float matrix[3][3], UINT8 vector[3], float result[3]) {
  for(int i=0; i<3; i++)
    result[i] = matrix[i][0]*vector[0] + matrix[i][1]*vector[1] + matrix[i][2]*vector[2];
}

inline void
_dot (float matrix[3][3], float vector[3], float result[3]) {
  for(int i=0; i<3; i++)
    result[i] = matrix[i][0]*vector[0] + matrix[i][1]*vector[1] + matrix[i][2]*vector[2];
}

inline void
_sub (Imaging im, float *matrix, float *result) {
  for(int y=0; y<im->ysize; y++) {
     for(int x=0; x<im->xsize; x++) {
        UINT8 *pixel = (UINT8*) &(im->image32[y][x]);
        result[(3 * y * im->xsize) + 3 * x] = pixel[0] - matrix[(3 * y * im->xsize) + 3 * x];
        result[(3 * y * im->xsize) + (3 * x) + 1] = pixel[1] - matrix[(3 * y * im->xsize) + (3 * x) + 1];
        result[(3 * y * im->xsize) + (3 * x) + 2] = pixel[2] - matrix[(3 * y * im->xsize) + (3 * x) + 2];
     }
  }
}

inline void
_add (Imaging im, float *matrix, float *result) {
  for(int y=0; y<im->ysize; y++) {
     for(int x=0; x<im->xsize; x++) {
        UINT8 *pixel = (UINT8*) &(im->image32[y][x]);
        result[(3 * y * im->xsize) + 3 * x] = pixel[0] + matrix[(3 * y * im->xsize) + 3 * x];
        result[(3 * y * im->xsize) + (3 * x) + 1] = pixel[1] + matrix[(3 * y * im->xsize) + (3 * x) + 1];
        result[(3 * y * im->xsize) + (3 * x) + 2] = pixel[2] + matrix[(3 * y * im->xsize) + (3 * x) + 2];
     }
  }
}

void
_write2output(float *buffer, Imaging im) {
  UINT8 r,g,b;  
  for(int y=0; y<im->ysize; y++) {
     for(int x=0; x<im->xsize; x++) {
        r = (UINT8) max(0, min(buffer[(3 * y * im->xsize) + (3 * x)], 255));
        g = (UINT8) max(0, min(buffer[(3 * y * im->xsize) + (3 * x) + 1], 255));
        b = (UINT8) max(0, min(buffer[(3 * y * im->xsize) + (3 * x) + 2], 255));
        im->image32[y][x] = r | g << 8 | b << 16 | b << 24;
     }
  }
}

int 
_daltonize(Imaging im, Imaging imOut, int mode) {
  // Daltonize image correction algorithm implemented according to
  // http://scien.stanford.edu/class/psych221/projects/05/ofidaner/colorblindness_project.htm

  // Transformation matrices for different color vision deficiencies
  float lms2lmsd[3][3] = {{1,0,0},{0.494207,0,1.24827},{0,0,1}};
  float lms2lmsp[3][3] = {{0,2.02344,-2.52581},{0,1,0},{0,0,1}};
  float lms2lmst[3][3] = {{1,0,0},{0,1,0},{-0.395913,0.801109,0}};
  // Colorspace transformation matrices
  float rgb2lms[3][3] = {{17.8824,43.5161,4.11935},{3.45565,27.1554,3.86714},{0.0299566,0.184309,1.46709}};
  float lms2rgb[3][3] = {{0.08094,-0.13050,0.116721},{-0.010249,0.054019,-0.113615},{-0.0003653,-0.00412161,0.6935}};
  // Daltonize image correction matrix
  float err2mod[3][3] = {{0,0,0},{0.7,1,0},{0.7,0,1}};
  // Requested color correction
  float lms2lms_deficit [3][3];
  
  if ((mode==1)||(mode==4))
     _initArray(lms2lms_deficit, lms2lmsd);
  else if ((mode==2)||(mode==5))
     _initArray(lms2lms_deficit, lms2lmsp);
  else if ((mode==3)||(mode=6))
     _initArray(lms2lms_deficit, lms2lmst);
  else
     return -1;

  Py_BEGIN_ALLOW_THREADS 

  // Transform to LMS space
  float *lms = NULL;
  lms = new float [im->xsize * im->ysize * 3];
  if (lms == NULL)
    return -1;
  
  for(int y=0; y<im->ysize; y++) {
     for(int x=0; x<im->xsize; x++) {
        UINT8 *pixel = (UINT8*) &(im->image32[y][x]);
        _dot(rgb2lms, pixel, &lms[(3 * y * im->xsize) + 3*x]);
	}
  }

  // Calculate image as seen with color vision deficiency
  float *_lms = NULL;
  _lms = new float [im->xsize * im->ysize * 3];
  if (_lms == NULL)
    return -1;
  
  for(int y=0; y<im->ysize; y++) {
     for(int x=0; x<im->xsize; x++) {
        float *pixel = &(lms[(3 * y * im->xsize) + 3*x]);
        _dot(lms2lms_deficit, pixel, &_lms[(3 * y * im->xsize) + 3*x]);
	}
  }

  float *_rgb = NULL;
  _rgb = new float [im->xsize * im->ysize * 3];
  if (_rgb == NULL)
    return -1;
  
  for(int y=0; y<im->ysize; y++) {
     for(int x=0; x<im->xsize; x++) {
        float *pixel = &(_lms[(3 * y * im->xsize) + 3*x]);
        _dot(lms2rgb, pixel, &_rgb[(3 * y * im->xsize) + 3*x]);
	}
  }

  // Check whether caculation of simulation is enough or whether daltonize is also requested
  if ( mode < 4) {
    // Calculate delta
    _sub(im, _rgb, lms); // Don't allocate too much memory: Reuse unused lms buffer

    // Do daltonize
    for(int y=0; y<im->ysize; y++) {
       for(int x=0; x<im->xsize; x++) {
          float *pixel = &(lms[(3 * y * im->xsize) + 3*x]);
          _dot(err2mod, pixel, &_lms[(3 * y * im->xsize) + 3*x]); // Don't allocate too much memory: Reuse unused _lms buffer
       }
    }   
    _add(im, _lms, _rgb);
  }

  // Return simulation or daltonized image
  _write2output(_rgb, imOut);

  free(_rgb);
  free(_lms);
  free(lms);

  Py_END_ALLOW_THREADS

  return 0;
}

/////////////////////////////////////////////////////////////////////////////
// Python callable functions
/////////////////////////////////////////////////////////////////////////////
static PyObject *
versions (PyObject *self, PyObject *args) {
  return Py_BuildValue ("sss", PILDALTONIZE, PYTHONVERSION, PILVERSION);
}

static PyObject *
about (PyObject *self, PyObject *args) {
  return Py_BuildValue("s", COPYRIGHTINFO);
}

static PyObject *
copyright (PyObject *self, PyObject *args) {
  return about(self, args);
}

static PyObject *
daltonize (PyObject *self, PyObject *args) {
  long idIn = 0L;
  long idOut = 0L;

  Imaging im = NULL;
  Imaging imOut = NULL;

  int mode = 0;
  int result = 0;

  if (!PyArg_ParseTuple(args, "lli", &idIn, &idOut, &mode)) {
    return Py_BuildValue("is", -1, "ERROR: Could not parse argument tuple, check documentation.");
  }

  im = (Imaging) idIn;
  imOut = (Imaging) idOut;

  // Only RGB images supported for now
  if (strcmp(im->mode, "RGB") != 0)
    return Py_BuildValue("is", -1, "ERROR: only RGB images supported.");

  result = _daltonize(im, imOut, mode);

  if (result == 0)
    return Py_BuildValue("is", 0, "");
  else
    return Py_BuildValue("is", result, "ERROR: unknown error occurred during daltonize operation.");

}

/////////////////////////////////////////////////////////////////////////////
// Python interface setup
/////////////////////////////////////////////////////////////////////////////
static PyMethodDef PILdaltonize_methods[] = {
  // pyCMS info
  {"versions", versions, 1, ".versions()"},
  {"about", about, 1, ".about()"},
  {"copyright", copyright, 1, ".copyright()"},

  {"daltonize", daltonize, 1, ".daltonize(imIn.im.id, imOut.im.id, mode)"},

  {NULL, NULL}
};


extern "C" {
  void initPILdaltonize(void)
  {
    Py_InitModule("PILdaltonize", PILdaltonize_methods);
  }
}
