# -*- coding: utf-8 -*-
"""
ipv4 - Internet protocol version 4 utilities

Internet address is a Python long. Functions taking Internet address
argument accept a long or a string using decimal notation. Functions
returning an Internet address return a long. To convert between the
decimal notation and number, use stringToAddress and addressToString.

Parts based on http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/66517

@copyright: Copyright © 2004 by Nir Soffer <nirs@freeshell.org>
@license: GNU GPL, see COPYING for details
"""

import socket
import struct


ADDRESS_MIN = 0L
ADDRESS_MAX = 0xffffffffL
ADDRESS_LENGTH = 4

def address(rep=0):
    """ Create address from various representations

    Actually creates a long, which represents an Internet address using
    IP version 4.

    example:
        >>> ipv4.address('192.114.134.51')
        3228730931L
        
    @param rep: address representation, either string or number
    @return: Internet address
    @rtype: long
    """

    if isinstance(rep, (unicode, str)):
        return stringToAddress(rep)
    
    if not isinstance(rep, (long, int)):
        raise TypeError('%s: address must be an integer' % repr(rep))
    if not (ADDRESS_MIN <= rep <= ADDRESS_MAX):
        raise ValueError('%s: address out of range' % repr(rep))
    return long(rep)


def stringToAddress(string):
    """ Convert dotted quad string to number 
    
    example:
        >>> ipv4.stringToAddress('127.0.0.1')
        2130706433L
    
    @param string: Internet address in decimal notation
    @return: Internet address
    @rtype: long
    """
    try:
        # ! for big endian byte order and standad size and alignment
        return struct.unpack('!L', socket.inet_aton(string))[0]
    except socket.error:
        raise ValueError('%s: invalid address' % str(string))
    

def addressToString(number):
    """ Convert number to dotted quad string 
    
    example:
        >>> ipv4.addressToString(0x7f000001L)
        '127.0.0.1'
        
    @param number: Internet address
    @return: Internet address in decimal notation
    @rtype: string
    """
    try:
        # ! for big endian byte order and standad size and alignment
        return socket.inet_ntoa(struct.pack('!L', number))
    except socket.error:
        raise ValueError('%s: invalid address' % str(number))
    

def mask(highbits):
    """ Create a 32 bit mask with highbits higher bits set 
    
    example:
        >>> ipv4.mask(24)
        4294967040L
        >>> ipv4.bin(ipv4.mask(24))
        '11111111111111111111111100000000'
    
    @param highbits: number of higher bits to set 0-32
    @return: ipv4 mask
    @rtype: long
    """
    if not  0 <= highbits <= 32:
        raise ValueError('%s: out of range ipv4 mask high bits' % repr(highbits))
    return ((1L << highbits) - 1) << (32 - highbits)
    
        
def netblock(rep='0.0.0.0/32'):
    """ Convert netblock from various representations

    Create from IPv4 string representations:
        address mask -- '127.0.0.0 255.255.255.0'
        address/mask-bits -- '127.0.0.0/24'
        
    or a sequence of addresses and mask, each one using any representation
    that address() can accept:
        (0x7f000001L , 0xffffff00L)
        ('127.0.0.0', '255.255.255.0')

    @param rep: string in various formats, see above
    @return: address range (min, max)
    @rtype: tuple
    """
    if isinstance(rep, (str, unicode)):
        rep = rep.strip()
        if '/' in rep:
            # CIDR format "127.0.0.0/24"
            ip, m = rep.split('/')
            ip = address(ip)
            m = mask(long(m))
        else:
            # Old format "127.0.0.0 255.255.255.0"
            ip, m = map(address, rep.split())            
    elif isinstance(rep, (tuple, list)):
        ip, m = map(address, rep)
    else:
        raise ValueError('%s: invalid netblok' % str(rep))
        
    min = ip & m
    max = min | (~m & ADDRESS_MAX)
    return min, max
       

def bin(n):
    """ return binary representation 

    Useful for visualizing bitwise operations.
    
    @param n: address to convert
    @return: pretty printed binary number
    @rtype: string
    """
    # Compute bits
    bits = []
    mask = 1L
    while 1:
        bits.append('01'[(n & mask) == mask])
        mask = mask << 1
        if mask > n: break
    bits.reverse()
    bits = ''.join(bits)
    bits = bits.zfill(ADDRESS_LENGTH * 8)        
        
    # Format as easier to read words: ... 10000000 00000001 
    words = []
    for i in range(0, len(bits), 8):
        words.append(bits[i:i+8])
    bits = ' '.join(words)
    return bits    
