# -*- coding: utf-8 -*-
"""
Generic test package generator

Copy as is to any directory containing test modules to make it a test package.

Naming rules:

 * Module name must start with 'test_' e.g.
 * Each test case class must end with 'TestCase' 
 * Each test function must start with 'test'

Modules are sorted by case insensitive compare, then a suite is made
from each test case class and added to the package suite. Test are run
in the order added to the suite.

Usage:
    from whatever import testpackage
    testpackage.runTests()

Based on MoinMoin._tests package copyright 2002-2004 by Jrgen Hermann <jh@web.de>

@copyright: Copyright © 2004 by Nir Soffer <nirs@freeshell.org>
@license: GNU GPL, see COPYING for details
"""

import os
import re
import unittest


def suite():
    """ Return test suite with all tests in testModules """

    # Get all test modules, starting with 'test_'
    testModules = [name[:-3] for name in os.listdir(os.path.dirname(__file__))
                   if name.startswith('test_') and name.endswith('.py')]
        
    # Sort test modules names by case insensitive compare to get nicer output
    caseInsensitiveCompare = lambda a, b: cmp(a.lower(), b.lower())
    testModules.sort(caseInsensitiveCompare)

    # Collect test suites from each test module
    suite = unittest.TestSuite()
    for module in testModules:
        # Import module
        module = __import__(__name__ + '.' + module, 
        	globals(), locals(), ['__file__'])
        # Collect TestCase classes and create suite for each one
        testCases = [unittest.makeSuite(obj, 'test') 
            for name, obj in module.__dict__.items()
            if name.endswith('TestCase')]
        suite.addTest(unittest.TestSuite(testCases))
    return suite


def runTests():
    """ Run test suite """
    unittest.TextTestRunner(verbosity=2).run(suite())


if __name__ == '__main__':
    runTests()
