# -*- coding: utf-8 -*-
"""
ipv4 tests

@copyright: Copyright  2004 by Nir Soffer <nirs@freeshell.org>
@license: GNU GPL, see COPYING for details
"""

import unittest
import time
import sys

from blacklist import ipv4
IP = ipv4.address


class CreateAddressTestCase(unittest.TestCase):
    def testCreateDefault(self):
        """ ipv4: create address with no value creates 0.0.0.0 """    
        self.assertEqual(IP(), 0L)
 
    def testCreateFromStrings(self):
        """ ipv4: create address from dotted quad string """    
        self.assertEqual(IP('127.0.0.1'), 0x7f000001L)
 
    def testCreateFromUnicode(self):
        """ ipv4: create address from dotted quad unicode """    
        self.assertEqual(IP(u'127.0.0.1'), 0x7f000001L)
     
    def testCreateFromNumber(self):
        """ ipv4: create address from int or long """
        self.assertEqual(IP(0x7f000001L), 0x7f000001L)
        self.assertEqual(IP(1), 1L)
    
    def testInvalidString(self):
        """ ipv4: create address with invalid string value raises ValueError """
        self.assertRaises(ValueError, IP, '127.0.0.')
        self.assertRaises(ValueError, IP, '127.0.0.0.1')
        self.assertRaises(ValueError, IP, '127.0.0.256')

    def testOutOfRange(self):
        """ ipv4: create address with out of range value raises ValueError """
        self.assertRaises(ValueError, IP, ipv4.ADDRESS_MIN - 1)
        self.assertRaises(ValueError, IP, ipv4.ADDRESS_MAX + 1)

    def testInvalidType(self):
        """ ipv4: create address with invalid argument type raises TypeError """
        self.assertRaises(TypeError, IP, 1.1)
        self.assertRaises(TypeError, IP, NotImplemented)
        self.assertRaises(TypeError, IP, None)


class ConvertTestCase(unittest.TestCase):
    def testAddressToString(self):
        """ ipv4: convert from address to string """
        self.assertEqual(ipv4.addressToString(0x7f000001L), '127.0.0.1')

    def testStringToAddress(self):
        """ ipv4: convert from string to address """
        self.assertEqual(ipv4.stringToAddress('127.0.0.1'), 0x7f000001L)
            
    def testBin(self):
        """ ipv4: convert address to easy to read binary number """    
        self.assertEqual(ipv4.bin(0), '00000000 00000000 00000000 00000000')
        self.assertEqual(ipv4.bin(255), '00000000 00000000 00000000 11111111')
        self.assertEqual(ipv4.bin(0xf000000fL), '11110000 00000000 00000000 00001111')


class NetblockTestCase(unittest.TestCase):
    def testCreateDefault(self):
        """ ipv4: create netblock with no arguments creates (0L, 0L) """
        self.assertEqual(ipv4.netblock(), (0L, 0L)) 

    def testCreateWithOldFormat(self):
        """ ipv4: create netblock from old format '127.0.0.0 255.255.255.0' """
        self.assertEqual(ipv4.netblock('127.0.0.0 255.255.255.0'),
                         (0x7f000000L, 0x7f0000ffL))

    def testCreateWithCIDRFormat(self):
        """ ipv4: create netblock from CIDR format '127.0.0.0/24' """
        self.assertEqual(ipv4.netblock('127.0.0.0/24'),
                         (0x7f000000L, 0x7f0000ffL))

    def testCreateWithStringSequence(self):
        """ ipv4: create netblock from string sequence ('127.0.0.0', '255.255.255.0') """
        self.assertEqual(ipv4.netblock(('127.0.0.0', '255.255.255.0')),
                         (0x7f000000L, 0x7f0000ffL))

    def testCreateWithNumberSequence(self):
        """ ipv4: create netblock from string sequence (0x7f000000L, 0xffffff00L) """
        self.assertEqual(ipv4.netblock((0x7f000000L, 0xffffff00L)),
                         (0x7f000000L, 0x7f0000ffL))

    def testInvalidNetblock(self):
        """ ipv4: invalid netblock raise ValueError """
        self.assertRaises(ValueError, ipv4.netblock, '10.0.0.0/45')
        self.assertRaises(ValueError, ipv4.netblock, '10.0.0./20')
        self.assertRaises(ValueError, ipv4.netblock, '10.0.0.0 256.255.255.0')
        self.assertRaises(ValueError, ipv4.netblock, '10.0.0.0 255.255.255.')

        
def suite():
    test_cases = [unittest.makeSuite(obj, 'test') 
        for name, obj in globals().items()
        if name.endswith('TestCase')]
    return unittest.TestSuite(test_cases)
    
if __name__ == '__main__':
    unittest.TextTestRunner(verbosity=2).run(suite())
