﻿# -*- coding: utf-8 -*-
"""
MoinMoin - SlideShow action

--------------------------------------------------------------------------------

Usage

    Allows treatment of a wiki page as a set of slides.  Displays a
    single slide at a time, along with a navigation aid.
    
    Use the GoToSlide macro to create a individual navigation on a slide,
    like a agenda or index

    A slide show page looks like this:

        general introduction or comments
        
        #Agenda 3
        #Author Dschingis Khan
        #Date 01. Jan 2008
        #Logo /path/to/a/image.png
        #Stylesheet projection
        
        = Slide 1 title =
        Slide 1 contents
        
        =  =
        Slide with empty title
        
        = Slide 3 title =
        Slide 3 contents
        ...

    The SlideShow action takes a parameter 'n', which is the (1-based)
    number of the slide to display.  The display uses the large screen
    'projection' media.
    
    Shortcuts:
     Exit: ESC-key
     Agenda: a
     Next slide: courser-key, spacebar
     Previous slide: courser-key
     
    Additional navigation:
     * a invisible area in the right bottom corner there mouse clicks move
       to the next page
     * pulldown list with slide headers
     * navigation footer
 
--------------------------------------------------------------------------------

Configuration

    Add the following lines to your wikiconfig.py if you want to use icons
    for your navigation:
    
    newsmileys = {
                  "(slideshow)":   (16, 16, 0, "moin-slide.png"),
                  "(first)":   (16, 16, 0, "moin-first.png"),
                  "(next)":   (16, 16, 0, "moin-next.png"),
                  "(exit)":   (16, 16, 0, "moin-exit.png"),
                  "(prev)":   (16, 16, 0, "moin-prev.png"),
                  "(autoplay)":   (16, 16, 0, "moin-play.png"),
                  "(stop)" :   (16, 16, 0, "moin-stop.png"),
                  "(edit)" :   (16, 16, 0, "moin-edit.png"),
                  "(index)" : (16, 16, 0, "moin-index.png"),
                  "(print)" : (16, 16, 0, "moin-print.png"),
                  "(goto)" : (16, 16, 0, "moin-goto.png"),
                  "(info)" : (16, 16, 0, "moin-info.png"),
                  "(close)" : (16, 16, 0, "moin-cancel.png"),
                  "(blank)" : (16, 16, 0, "blank.gif"),
                  }

    config.smileys.update( newsmileys )
    
--------------------------------------------------------------------------------

History
    
    1.0 2005-03-25 - first version
    
    1.1 2005-03-30 - revised thanks to some comments from NirSoffer.
    
     * Parse H1 tags instead of using the [[Slide]] macro.
     * Improve navigation.
    
    1.2 2005-03-31 rewrite by NirSoffer:
     
     * fix bug with heading within pre,
     * simplifiying code
     * auto correct slide number
     * much more
     
    1.3 2005-04-04 - complete rewrite by Nir Soffer
    
     * Refactor into 2 classes, Slides and SlideShow
     * Easy to customize design, subclass and override
       few methods to customize behavior.
     * Parse page headers and set pragmas
     * Disable conflicting section-number pragma
     * Support any headers e.g #Author, #Status
     * When slide number out of range, default to first slide
     * Fixed bug when size of navigation changed
     * Complete abstract output using the formatter
     * Added required Dependencies variable
     * Simplified more efficient parsing
     * More efficient link creation
     * Removed unneeded last slide link
     * Improved navigation tooltips
     * Default slide for pages with no heading
     * Complete right to left support - both back forward icons and
       hotkeys are switched according to the user language.
     * Disable link to editor if you can't edit
     * Clear text links instead of funny text mode icons.
     * Changed hotkeys to up, right and left arrow, which make sense and
       work everywhere. As a benefit, space and numbers can be use on
       interactive slides.
       
    1.4 2008-01-04 - add feathers from todo list by Richard Flieger
     
     * Icons for "edit", "exit", "first", "back" and "previous" links.
     * Additional links for "help", "agenda"
     * Icons for "help" and "agenda"
     * Additional hotkey for "exit", "help" and "agenda"
     * Selectbox for easy navigation by slide headers
     * Hide the navigation panel on mouseout
     * Show next slide by clicking in the right-bottom corner
     * Additional user defined tags:
       * #Agenda
        Number of the slide with the agenda
       * #Author
        Name of the author
       * #Date
        Date of the slideshow
       * #Logo
        Path of an attachment
       * #Stylesheet
        Name of the stylesheet
     * Show agenda in a popupbox.
     * Show this help in a popupbox.

--------------------------------------------------------------------------------

TODO:

     * Use stylesheet specified on page with #stylesheet pi
     * Have slide show style sheets directory in htdocs - slide show should not
       use theme stylesheets but the slideshow stylesheet. You can use modern
       as your theme and display slide show using another design.
     * Old style one-slide-per-page are not displayed nicely by stylesheet 
       optimized to this action, as the heading is not rendered in a separate
       heading div.
     * SlideShow macro to view any slide on any page
     * A way to view any slide from a standard page view, maybe use
       slideshow  parser - sub class wiki parser to add links to slides
       on headings?
     * Easy to use popup menu with all available slideshow styles,
       accessible from the navigation pannel?
     * Move script into .js file?
     * Javascript transitions??
     * Create the html head links in the action, so we can use correct
       <link rel="Start" href="..."> and like.
     * Same for stylesheet links, so we can control the stylesheet
     * A way to have edit mode and presentation mode navigation?
     * A way to edit the current slide in a popupbox
     
    
    @copyright: 2005 Nir Soffer <nirs@freeshell.org>
    @copyright: 2005 Jim Clark
    @copyright: 2008 Richard Flieger
    @license: GNU GPL, see COPYING for details.
"""

import re
from StringIO import StringIO
from MoinMoin import config, wikiutil, i18n
from MoinMoin.util import web
from MoinMoin.Page import Page
from MoinMoin.formatter.text_html import Formatter
from MoinMoin.action import AttachFile

Dependencies = ['language']
        
    
class Slides(object):
    """ A collection of slides saved in a wiki page """
    
    def __init__(self, request, page):
        self._request = request
        self._page = page
        self._headers = {}
        self._slides = []
        self._parserClass = None
        self._isEditable = None

        # Parse page body        
        self._parseHeaders()
        self._parseSlides()

    #-- Quering slideshow
    
    def name(self):
        return self._page.split_title(self._request, force=1)
        
    def stylesheet(self):
        return self._headers.get('Stylesheet', 'projection')

    def language(self):
        return self._headers.get('language', self._request.lang)
    
    def parserClass(self):
        if not self._parserClass:
            # Ignore the page parser and use always wiki parser
            ## name = self._headers.get('format', 'wiki')
            name = 'wiki'
            Parser = wikiutil.importPlugin(self._request.cfg, 'parser', name, 
                                           'Parser')
            if Parser is None:
                # default to plain text formatter (i.e. show the page source)
                from MoinMoin.parser.plain import Parser
            self._parserClass = Parser
        return self._parserClass
    
    def __len__(self):
        return len(self._slides)
    
    def isEmpty(self):
        return len(self) == 0
    
    def isEditable(self):
        if not self._isEditable:
            self._isEditable = self._request.user.may.write(self._page.page_name)
        return self._isEditable
    
    #-- Accssing slides
    
    def __getitem__(self, index):
        return self.titleAt(index), self.bodyAt(index)

    def titleAt(self, index):
        return self._slides[index - 1].group('text').strip()
        
    def bodyAt(self, index):
        start = self._slides[index - 1].end('heading')
        try:
            end = self._slides[index].start('heading')
            return self._body[start:end]
        except IndexError:
            return self._body[start:]        
        
    #-- Headers parsing
    
    # This generic code is not really related to slideshow and should be
    # moved into page class (replacing send_page).
    
    _headerScanner = re.compile(r'^#(?P<key>[^#]+?)\s+(?P<value>.+?)\s*$',
                                re.MULTILINE | re.UNICODE)
    
    def _parseHeaders(self):
        for match in self._headerScanner.finditer(self._page.getPageHeader()):
            key, value = match.group('key'), match.group('value')
            setter = getattr(self, '_set_' + key, None)
            if setter:
                setter(value)
            else:
                self._headers[key] = value
    
    def _set_language(self, value):
        # Ignore language the wiki does not know
        if value in i18n.wikiLanguages():
            self._headers['language'] = value
            
    def _set_pragma(self, value):    
        try:
            key, value = value.split(' ', 1)
            self._request.setPragma(key, value)
        except (ValueError, TypeError):
            pass # Ignore invalid pragmas

    #-- Slide parsing
    
    # Must skip headings within pre or code area
    # TODO: works with wiki markup only - should use the page parser
    _slideScanner = re.compile(r"""
        (?P<skip>{{{(?:.*\n)+?}}}) |                # skip this
        (?:^\s*(?P<heading>=\s(?P<text>.*)\s=)$)     # match this
        """, re.MULTILINE | re.UNICODE | re.VERBOSE)
    
    def _parseSlides(self):
        self._body = self._page.getPageText()
        for match in self._slideScanner.finditer(self._body):
            if match.start('skip') != -1:
                continue
            self._slides.append(match)
                        

class SlideShow:
    """ Show single slide from a page """

    action = 'SlideShow'
    
    #-- Public methods

    def __init__(self, request, pagename):
        # General action code, move to Action base class
        self.request = request
        self.pagename = pagename
        self.page = request.page = Page(request, pagename)
        self.formatter = request.formatter = Formatter(request)
        self.formatter.page = self.page
        self.theme = request.theme

        # SlideShow specific init
        self.slides = Slides(self.request, self.page)
        # Disable section-numbers - a slide can have only one section.
        request.setPragma('section-numbers', 'off')
        try:
            self.current = int(request.form.get('n', [1])[0])
            self.normalizeCurrent()
        except ValueError:
            self.current = 1
        self.maxSlideLinks = 15        
        
        # Lazy initialized values that does not change during request.
        self._urlToPageTemplate = None
        self._urlToSlideTemplate = None
        self._userMayEdit = None

    def render(self):
        """ Render current slide """
        
        request = self.request
        _ = request.getText
        
        request.http_headers()
        
        # Slide content uses the slideshow language
        request.setContentLanguage(self.slides.language())
        wikiutil.send_title(request, self.slides.name(), page=self.page,
                            print_mode=1, media='projection')     
        request.write(self.logo())
        request.write(self.title())
        request.write(self.author())
        request.write(self.date())
        
        
        if self.slides.isEmpty():
            request.write(self.defaultSlide())
        else:
            request.write(self.heading())
            request.write(self.content())
        
        # Navigation uses the user language
        self.request.setContentLanguage(self.request.lang)
        request.write(self.navigationScript())
        request.write(self.navigation())
        
        wikiutil.send_footer(request, self.pagename, print_mode=1)

    #-- User input handling 
    
    def normalizeCurrent(self):
        """ Default silently to first slide if value is out of range """
        if not 1 <= self.current <= len(self.slides):
            self.current = 1

    #-- Link creation

    def urlToPage(self, query=None):
        if not self._urlToPageTemplate:
            self._urlToPageTemplate = self.page.url(self.request)
        if query:
            return '%s?%s' % (self._urlToPageTemplate, query)
        return self._urlToPageTemplate
        
    def urlToSlide(self, index):
        """ Return url to slide at index """
        if not self._urlToSlideTemplate:
            self._urlToSlideTemplate = (self.urlToPage('action=%s&n=' % 
                                                        self.action))
        return self._urlToSlideTemplate + str(index)


    def linkTo(self, url, text, attributes=None, smiley="", icon=""):
        """ Don't call directly, use one of the linkToX methods """
        if attributes is None:
            attributes = {}
        if smiley:
            try:
                smiley = self.formatter.smiley(smiley)
            except:
                pass
        if icon:
            try:
                icon = self.formatter.icon(icon)
            except:
                pass
        text = self.formatter.text(text)
        if smiley or icon:
            text = ""
        items = [self.formatter.url(1, url, **attributes),
                 smiley,
                 icon,
                 text,
                 self.formatter.url(0),]
        return ''.join(items)

    def linkToSlide(self, index, text=None, attributes=None, smiley="", icon=""):
        if text is None:
            text = unicode(index)
        if index == self.current:
            return self.disabledLink(text, attributes, smiley=smiley, icon=icon)
        return self.linkTo(self.urlToSlide(index), text, attributes, smiley=smiley, icon=icon)
    
    def linkToEditor(self, text, attributes=None):
        if self.slides.isEditable():
            attributes = { "title" : text,'onmouseover': "navigation('show')" }
            return self.linkTo(self.urlToPage('action=edit'), text, attributes, smiley="(edit)")
        return self.disabledLink(text)
        
    def linkToPage(self, text=None, attributes=None):
        if text is None:
            text = self.page.page_name
        return self.linkTo(self.urlToPage(), text, attributes, smiley="(exit)")

    def disabledLink(self, text, attributes=None, smiley="", icon=""):
        cssClass = attributes and attributes.get('class')
        if smiley:
            try:
                smiley = self.formatter.smiley(smiley)
            except:
                pass
        if icon:
            try:
                icon = self.formatter.icon(icon)
            except:
                pass
        text = self.formatter.text(text)
        if smiley or icon:
            text = ""
        if cssClass:
            # Use code_token as a hack to get a span
            items = [self.formatter.code_token(1, cssClass),
                     smiley,
                     icon,
                     text,
                     self.formatter.code_token(0, cssClass),]
            return ''.join(items)
        return self.formatter.text(text) + smiley + icon

    #-- Adpating to language direction
    
    def adaptToLanguage(self, direction):
        # In RTL, directional items should be switched
        if i18n.getDirection(self.request.lang) == 'rtl':
            return not direction
        return direction
    
    def forwardIcon(self, forward=True):

        forwardsymbol = u'\u2190'
        backsymbol = u'\u2192'
        return [forwardsymbol, backsymbol][self.adaptToLanguage(forward)]
    
    def backIcon(self):
        return self.forwardIcon(False)

    def forwardArrowKey(self, forward=True):
        rightArrowKey = 39
        leftArrowKey = 37
        return [leftArrowKey, rightArrowKey][self.adaptToLanguage(forward)]
        
    def backArrowKey(self):
        return self.forwardArrowKey(False)
  
    #-- Rendering default slide
  
    def defaultSlide(self):
        """ Rendered for pages without headings """
        _ = self.request.getText
        items = [
                 self.formatter.paragraph(1),
                 self.linkToEditor(_('Edit page to add slides')),
                 self.formatter.paragraph(0),]
        return ''.join(items)

    #-- Rendering current slide

    def logo(self):
        """ Return formatted logo """
        image = self.slides._headers.get("Logo","")
        items = []
        if image:
            kw = {}
            pagename, attname = AttachFile.absoluteName(image, self.formatter.page.page_name)
            kw['src'] = AttachFile.getAttachUrl(pagename, attname, self.request)
            items = [
                 self.formatter.startContent("logo"),
                 self.formatter.url(1),
                 self.formatter.image(**kw),
                 self.formatter.url(0),
                 self.formatter.endContent(),]
            return ''.join(items)
    
    def title(self):
        """ Return formatted title """
        title = self.slides._headers.get("Title","")
        items = []
        if title:
            items = [
                 self.formatter.startContent("title"),
                 self.formatter.text(title),
                 self.formatter.endContent(),]
            return ''.join(items)

    def author(self):
        """ Return formatted name of the author """
        auhtor = self.slides._headers.get("Author","")
        items = []
        if auhtor:
            items = [
                 self.formatter.startContent("author"),
                 self.formatter.text(auhtor),
                 self.formatter.endContent(),]
            return ''.join(items)

    def date(self):
        """ Return formatted date """
        date = self.slides._headers.get("Date","")
        items = []
        if date:
            items = [
                 self.formatter.startContent("date"),
                 self.formatter.text(date),
                 self.formatter.endContent(),]
            return ''.join(items)

    def heading(self):
        """ Return formatted heading """
        items = []
        if self.slides.titleAt(self.current):
            items = [
                 self.formatter.startContent("heading"),
                 self.formatter.heading(1, 1),
                 self.formatter.text(self.slides.titleAt(self.current)),
                 self.formatter.heading(0, 1),
                 self.formatter.endContent(),]
        return ''.join(items)

    def content(self):
        """ Return formatted body """
        content = StringIO()
        self.request.redirect(content)
        Parser = self.slides.parserClass()
        Parser(self.slides.bodyAt(self.current), 
               self.request).format(self.formatter)
        self.request.redirect()
        
        items = [self.formatter.startContent("content"),
                 content.getvalue(),
                 self.formatter.endContent(),]
        return ''.join(items)
    
    def popUpBox(self,boxtitle="popupbox",boxid="",content="",smiley="",attributes={}):
        _ = self.request.getText
        if not boxid:
            boxid = boxtitle.rstrip().lower()
        onclick =  {'onclick': boxid + "('toogle')"}
        boxtitle = "<h2>" + boxtitle + "</h2>"
        attributes.update(onclick)
        icon = self.formatter.smiley(smiley)
        closebutton = self.linkTo(url="#", text=_("Close"),attributes=onclick, smiley="(close)")
        boxdict =  {
                    u"boxtitle" : boxtitle,
                    u"boxid" : boxid,
                    u"boxcontent" : content,
                    u"icon" : icon,
                    u"closebutton" : closebutton,
                    }
  
        boxscript = """
<script language="JavaScript" type="text/javascript">

function %(boxid)s(showhide)
{
  if(document.getElementById('%(boxid)s').style.visibility == "hidden")
  {
   document.getElementById('%(boxid)s').style.visibility="visible";
  }
  else
  {
   document.getElementById('%(boxid)s').style.visibility="hidden";
  }
}
</script>"""

        boxhtml = """

<div id="%(boxid)s" class="%(boxid)s"  style="visibility:hidden;" onmouseout="navigation('show')">
 <div calss="popupboxheader">
  <div class="popupboxicon" ondblclick="%(boxid)s('toogle')"> %(icon)s </div>
  <div class="popupboxtitle" ondblclick="%(boxid)s('toogle')"> %(boxtitle)s </div>
  <div class="popupboxclosebutton" ondblclick="%(boxid)s('toogle')"> %(closebutton)s </div>
 </div>
 <div class="popupboxcontent" ondblclick="%(boxid)s('toogle')"> %(boxcontent)s </div>
</div>
"""

        box = u''.join([boxscript,
                        boxhtml])
        
        link = self.linkTo(url="#", text=boxtitle,attributes=attributes, smiley=smiley)
        return link,box % boxdict
    
    #-- Rendering navigation interface
    
    def navigationScript(self):
        """ Return javascript keydown event handler """
        return """
<script type="text/javascript">
<!--// SlideShow functions
function getKey(e) {
    // Support multiple browsers stupidity
    var key;
    if (e == null) {
        // IE
        key = event.keyCode;
    } else { 
        // Standard complaient
        if (e.altKey || e.ctrlKey) {
            return null;
        }
        key = e.which;
    }
    return key;
}
function go(slide) {window.location="%(base)s" + slide;}
function gotourl(url) {window.location=url;}
function goforward() { go('%(previous)d'); }
function onkeydown(e) {
    switch(getKey(e)) {
        case 36: go('1'); break; // home
        case %(back)d: go('%(next)d'); break; // back arrow
        case 32: go('%(previous)d'); break; // spacebar
        case 75: go('%(previous)d'); break; // left click
        case 27: gotourl("%(urltopage)s"); break; // esc 
        case %(forward)d: go('%(previous)d'); break; // forward arrow
        default: return true; // pass event to browser
    }
    // Return false to consume the event
    return false;
}
document.onkeydown = onkeydown
//-->
</script>
""" % {'base': self.urlToSlide(''), # slide number added by javascript
       'next': self.current - 1,
       'previous': self.current + 1,
       'back': self.backArrowKey(),
       'forward': self.forwardArrowKey(),
       'urltopage' : self.urlToPage(),
       }

    def slideLinksRange(self):
        """ Return range of slides to display, current centered """
        other = self.maxSlideLinks -1 # other slides except current
        start = max(1, self.current - other / 2)
        end = min(start + other, len(self.slides))
        start = max(1, end - other)
        return range(start, end + 1)
     
    def addLinksBeforeSlides(self, items):
        _ = self.request.getText
        add = items.append
        
        add(self.linkToEditor(_('Edit'), 
                              attributes={'title': _('Edit slide show'),'onmouseover': "navigation('show')"}))
        add(self.linkToPage(_('Quit'), 
                            attributes={'title': _('Quit slide show'),'onmouseover': "navigation('show')"}))
        add(self.linkToSlide(1, text=_('Start'),
            attributes={'title': _('Show first slide (up arrow)'),'onmouseover': "navigation('show')"},smiley="(first)"))

    def addLinksToSlides(self, items):
        _ = self.request.getText
        items.append(self.linkToSlide(max(self.current - 1, 1), text=self.backIcon(), 
            attributes={'title': _('Show previous slide (back arrow)'),'onmouseover': "navigation('show')"},smiley="(prev)"))
        for i in self.slideLinksRange():
            attributes = {'title': self.slides.titleAt(i)}
            if i == self.current:
                attributes = {'class': 'current','onmouseover': "navigation('show')"}
            items.append(self.linkToSlide(i, attributes=attributes))
            
        index = min(self.current + 1, max(len(self.slides), 1))
        items.append(self.linkToSlide(index, text=self.forwardIcon(), 
                     attributes={'title': _('Show next slide (forward arrow)')},smiley="(next)"))

        url = self.urlToPage('action=%s&n=' % self.action)

    def addSlideList(self, items):
        menutemplate = u"""

        <form OnMouseOver="navigation('show')" class="slidemenu" method="get" action="" style="display:inline">
        <input name="action" value="SlideShow" type="hidden">
        <select name="n" OnMouseOver="navigation('show')" onchange="if ( (this.options[this.selectedIndex].disabled == false)) {this.form.submit();
            }
            this.selectedIndex = 0;" style="border:thin solid #E5E5E6;margin:0px 0px 1px 0px">
        %(entrylist)s
            </select>
    </form>"""
        entrytemplate = u"""<option %(selected)s value="%(slideindex)s">%(slidetitle)s</option>"""
        entrylist = []
        
        _ = self.request.getText
        i = 0

        for i in range(len(self.slides)):
            
            selected = ""
            if i + 1 == self.current:
                selected = u"selected"

            entrylist.append(entrytemplate % { u"slideindex" : str(i+1),
                                               u"slidetitle" : self.formatter.text(wikiutil.escape(self.slides.titleAt(i+1),1)),
                                               u"selected" : selected } )
            
        menu = menutemplate % { u"entrylist" : u"\n".join(entrylist) }
        items.append(menu)
        
    def addPopupHelp(self, items):
        _ = self.request.getText
        help = re.search( "^Usage$^.*", __doc__, re.M|re.I )
        if help != None:
            help = help.group( 0 )
        else:
            help = ""
        help = __doc__
        help = "{{{" + help + "}}}"
        link, popupbox = self.popUpBox(boxtitle=_("Help"), content=self.getHTML(help),boxid="help", attributes={'title': _('Show slide help'),'onmouseover' : "navigation('show')"},smiley="(info)")
        items.append(link + popupbox)
        return items

    def addPopupAgenda(self, items):
        _ = self.request.getText
        script = """
        <script type="text/javascript">
        function getKey(e) {
    // Support multiple browsers stupidity
    var key;
    if (e == null) {
        // IE
        key = event.keyCode;
    } else { 
        // Standard complaient
        if (e.altKey || e.ctrlKey) {
            return null;
        }
        key = e.which;
    }
    return key;
}

function onkeydown(e) {
    switch(getKey(e)) {
            case 65: agenda('toogle'); break;
        default: return true; // pass event to browser
    }
    // Return false to consume the event
    return false
}
</script>"""
        agenda = self.getAgenda()
        link, popupbox = self.popUpBox(boxtitle=_("Agenda"), content=agenda,boxid="agenda", attributes={'title': _('Show agenda'),'onmouseover' : "navigation('show')"},smiley="(index)")
        items.append(script)
        items.append(link + popupbox)
        return items
    
    # TODO
    def addLinkToPrint(self, items):
        # currently not implemented
        _ = self.request.getText
        printbutton = self.linkTo(url="#", text=_("Print"), smiley="(print)")
        items.append(printbutton)
        return items
    
    def getAgenda(self):
        agenda = self.slides._headers.get("Agenda")
        try:
            agenda = int(agenda)
        except:
            return ""
        content = self.slides.bodyAt(agenda)
        return self.getHTML(content)
    
    def getHTML(self,text):
        content = StringIO()
        self.request.redirect(content)
        Parser = self.slides.parserClass()
        Parser(text,self.request).format(self.formatter)              
        self.request.redirect()
        return content.getvalue()
    
    def addLinksAfterSlides(self, items):
        _ = self.request.getText
    
    def navigationItems(self):
        items = []
        self.addPopupAgenda(items)
        self.addPopupHelp(items)
        # TODO
        #self.addLinkToPrint(items)         # currently not implemented
        self.addLinksBeforeSlides(items)
        self.addSlideList(items)
        self.addLinksToSlides(items)
        self.addLinksAfterSlides(items)
        return items
        
    def navigation(self):
        """ Return navigation section """
        formatter = self.formatter
        boxscript = """
        <script language="JavaScript" type="text/javascript">
  function navigation(showhide){
    if(showhide == "show"){
        document.getElementById('navigation').style.visibility="visible";
    }else if(showhide == "hide"){
        document.getElementById('navigation').style.visibility="hidden";
    }
  }
  </script>
        """
        boxhtml = """<div OnMouseOut="navigation('hide')" id="navigation" style="visibility: hidden;">
%(content)s
</div>"""
        content =  ''.join([formatter.listitem(1) + i + formatter.listitem(0)
                          for i in self.navigationItems()])
        
        content = ''.join([formatter.startContent("navigationbar"),
                          formatter.bullet_list(1),
                           content,
                           #formatter.listitem(1) + """<a href="javascript:navigation('hide');">close</a>""" + formatter.listitem(0),
                           formatter.bullet_list(0),
                           formatter.endContent()])
        
        items = [
                 formatter.rawHTML(boxscript),
                 formatter.rawHTML(boxhtml % { u"content" : content} ),
                 formatter.startContent("mousenavigationbar"),
              """<div OnMouseOver="navigation('show')"><font style="visibility: hidden;">.</font></div>""",
                 formatter.endContent(),
                 formatter.startContent("mousenextslidebar"),
              """<div onclick="goforward()"><font style="visibility: hidden;">.</font></div>""",
                 formatter.endContent(),
                 ]
        return ''.join(items) 

def execute(pagename, request):
    """ Glue to current plugin system """
    request.clock.start('slideshow')
    SlideShow(request, pagename).render()
    request.clock.stop('slideshow')
